########################################################################
# Processes the saved preferences for the user.  These preferences
#   include RS232 settings, directory settings, and floppy-type.
#   Format of the ini file is like the Windows versions with a section
#   followed by values.  Writing involves naming a section, an attribute,
#   and a value.  Nothing really new here.
#
# This is implemented as an internal set of 2 arrays that hold the values
#   for each section as a list.  Each element of the list is an attribute
#   name with NO SPACES followed by an attribute value with anything.
########################################################################

# NOTE - one default that MUST be set is "iniFile" which specifies
#        the filename of the file that has the attributes.


########################################################################
# setIniDefaults() -	Set up the default .ini values that will be used
#                       if not otherwise overridden by the .ini file
#                       itself.
########################################################################
proc setIniDefaults {} {
    global home
    global tcl_platform

    #
    # now iniailze the defaults themselves
    #

    iniSetDefault "Program" "iniFile"         "svd.ini"
    iniSetDefault "Program" "iniBackupFile"   "svd.inb"

    iniSetDefault "Program" "tipsFile"        "tips.ini"

    iniSetDefault "Program" "OSListDir"       "OSFiles"
    iniSetDefault "Program" "OSListExtension" "osl"

    iniSetDefault "Program" "AutoProbe"           1
    iniSetDefault "Program" "AutoLoad"            0
    iniSetDefault "Program" "AutoReset"           1
    iniSetDefault "Program" "H8Swap"              1
    iniSetDefault "Program" "SectorUpload"        0
    iniSetDefault "Program" "GenericFile"         1

    iniSetDefault "Program" "Position"        "+10+10"

    if { $tcl_platform(platform) == "windows" } {
	iniSetDefault "Font" "TipsWidth"          "3i"
	iniSetDefault "Font" "Tips"               "Helvetica 8"
	iniSetDefault "Font" "TipsTitle"          "Helvetica 10 bold italic"
	iniSetDefault "Font" "Message"            "Helvetica 10 bold"
	iniSetDefault "Font" "ProgramOutput"      "Courier 8"
    } else {    
	iniSetDefault "Font" "TipsWidth"          "4.5i"
	iniSetDefault "Font" "Tips"               "Helvetica 12"
	iniSetDefault "Font" "TipsTitle"          "Helvetica 13 bold italic"
	iniSetDefault "Font" "Message"            "Helvetica 12 bold"
	iniSetDefault "Font" "ProgramOutput"      "Courier 10"
    }

    iniSetDefault "Floppy" "Type"             "generic"

# RS232 settings

    iniSetDefault "RS232" "Unix-Speed" "115200"
    iniSetDefault "RS232" "Unix-Device" ""
    iniSetDefault "RS232" "Windows-Speed" "57600"
    iniSetDefault "RS232" "Windows-Device" "com1:"

# Image directory settings

    iniSetDefault "Directories" "Unix-images-generic" "~"
    iniSetDefault "Directories" "Unix-files-generic" "~"
    iniSetDefault "Directories" "Windows-images-generic" "c:"
    iniSetDefault "Directories" "Windows-files-generic" "c:"

}

########################################################################
# iniSetAttribute() -	Sets the given attribute in the given section.
########################################################################
proc iniSetAttribute {section attribute value} {
    global   iniSections

    if { [array names iniSections -exact $section] == "" } {
	set iniSections($section) [list "$attribute $value"]
    } else {
	if { [iniCheckAttribute $section $attribute] == 2 } {
	    iniDelAttribute $section $attribute
	}
	set iniSections($section) [linsert $iniSections($section) end "$attribute $value"]
    }
}

########################################################################
# iniDelAttribute() -	Deletess the given attribute in the given section.
#                       This is normally used when a bad value is read
#                       (like one that is no longer supported) and you
#                       need to delete it then re-read the default.
########################################################################
proc iniDelAttribute {section attribute} {
    global   iniSections

    if { [array names iniSections -exact $section] == $section &&
         [set found [lsearch $iniSections($section) "$attribute *"]] != -1 } {
	     set iniSections($section) [lreplace $iniSections($section) $found $found]
    }
}

########################################################################
# iniSetDefault() -	Sets the default for a given attribute.
########################################################################
proc iniSetDefault {section attribute value} {
    global   iniDefaults

    if { [array names iniDefaults -exact $section] == "" } {
	set iniDefaults($section) [list "$attribute $value"]
    } else {
	set iniDefaults($section) [linsert $iniDefaults($section) end "$attribute $value"]
    }

}

########################################################################
# iniGetSection() -	Gets a whole section of attributes.
########################################################################
proc iniGetSection {section} {
    global  iniSections

    if { [array names iniSections -exact $section] == $section } {
	return [concat $iniSections($section) [iniGetDefaultSection $section]]
    } else {
	return [iniGetDefaultSection $section]
    }
}

########################################################################
# iniGetDefaultSection() -	Gets a whole section of attributes from
#                               the default side.
########################################################################
proc iniGetDefaultSection {section} {
    global   iniDefaults

    if { [array names iniDefaults -exact $section] == $section } {
	return $iniDefaults($section)
    } else {
	return
    }
}

########################################################################
# iniGetAttribute() -	Gets the given attribute from the attribute set.
#                       If the attribute isn't in the given section, a
#                       check for it is made in the default attribute
#                       set.  If found in either, it is returned.
#                       Otherwise, a blank string is returned.  You
#                       may want to use iniCheckAttribute() first to see
#                       if an attribute has been set (or defaulted) before
#                       retrieving it.
########################################################################
proc iniGetAttribute {section attribute} {
    global   iniSections

    if { [array names iniSections -exact $section] == $section &&
         [set found [lsearch $iniSections($section) "$attribute *"]] != -1 } {
	     regexp {^[^ ]* (.*)$} [lindex $iniSections($section) $found] dummy found
    } else {
	set found [iniGetDefaultAttribute $section $attribute]
    }

    return $found
}

########################################################################
# iniGetDefaultAttribute() -	Gets the given attribute's default from
#                               the attribute set.  If found, it is returned.
#                       Otherwise, a blank string is returned.  You
#                       may want to use iniCheckAttribute() first to see
#                       if an attribute has been defaulted before
#                       retrieving it.
########################################################################
proc iniGetDefaultAttribute {section attribute} {
    global   iniDefaults

    if { [array names iniDefaults -exact $section] == $section &&
         [set found [lsearch $iniDefaults($section) "$attribute *"]] != -1 } {
	     regexp {^[^ ]* (.*)$} [lindex $iniDefaults($section) $found] dummy found
    } else {
	    set found ""
    }
    return $found
}

########################################################################
# iniCheckAttribute() -	Returns 0 if an attribute doesn't exist.  Returns
#                       1 if it is a default, and 2 if it was explicitly
#                       set.
########################################################################
proc iniCheckAttribute {section attribute} {
    global iniSections
    global iniDefaults

    if { [array names iniSections -exact $section] == $section &&
         [lsearch $iniSections($section) "$attribute *"] != -1 } {
	return 2
    } else {
	if { [array names iniDefaults -exact $section] == $section &&
	     [lsearch $iniDefaults($section) "$attribute *"] != -1 } {
	    return 1
	} else {
	    return 0
	}
    }
    
}

########################################################################
# iniEnumerateSection() -	Enumerate all of the attributes in the
#                               given section.  Doesn't differentiate
#                               between defaults and set attributes.
#                          
#                               The first item is returned.
#     
#                               A cursor is returned that should be used
#                               for the next call to iniEnumerateNext().
#
#                               NOTE - a "snapshot" of the section is
#                               taken so that modifications to it won't
#                               affect these routines' operation.
########################################################################
proc iniEnumerateSection {section} {
    global iniSections
    global iniDefaults

    set ret ""

    if { [array names iniSections -exact $section] == $section } {
	set ret [concat $ret $iniSections($section)]
    }
    if { [array names iniDefaults -exact $section] == $section } {
	set ret [concat $ret $iniDefaults($section)]
    }

    return $ret
}

########################################################################
# iniWrite() -	Writes the ini file out to disk.  The old ini is written
#               to a backup file.
#
#               The name of the ini file is found in the defaults.  If
#               it is not there, then the file IS NOT written.
#
#               Return is 0 if NOT WRITTEN, 1 otherwise.
########################################################################
proc iniWrite {} {
    global   iniSections

    if { [iniGetAttribute "Program" "iniFile"] != "" } {

	if { [iniGetAttribute "Program" "iniBackupFile"] != "" } {

	    if { [file exists [iniGetAttribute "Program" "iniBackupFile"]] == 1 } {
		file delete [iniGetAttribute "Program" "iniBackupFile"]
	    }
	    if { [file exists [iniGetAttribute "Program" "iniFile"]] == 1 } {
		file rename [iniGetAttribute "Program" "iniFile"] [iniGetAttribute "Program" "iniBackupFile"]
	    }
	}

	if { [array exists iniSections] == 1 && [array size iniSections] != 0 } {
	    set outputfile [open [iniGetAttribute "Program" "iniFile"] w]

	    set searchid [array startsearch iniSections]
	    while { [set section [array nextelement iniSections $searchid]] != "" } {
		puts $outputfile [format {[%s]} $section]
		foreach attributevalue $iniSections($section) {
		    regexp {^([^ ]*) (.*)$} $attributevalue dummy attribute value
		    puts $outputfile [format {%s=%s} $attribute $value]
		}
	    }
	    array donesearch iniSections $searchid

	    close $outputfile
	}
	return 1
    } else {
	return 0
    }
}
########################################################################
# iniReadChannel() -	Given a channel, read in the attributes there.
#               If lines are read outside of sections, they are IGNORED!
#               0 is returned if it was NOT read, 1 otherwise.  If the
#               inifile is not read, that means either that the name of
#               the file is not known, or that the filename wasn't valid.
########################################################################
proc iniReadChannel {inputfile def} {

    set currentsection ""

    while { [gets $inputfile inputline] >= 0 } {

	if { [regexp {^\[(.*)\]$} $inputline allmatch newsection] == 1 } {
	    set currentsection $newsection
	} elseif { [regexp {^#.*$} $inputline commentline] == 1 } {
	    # comment lines are ignored...and LOST!
	} elseif { [regexp {^([^=]*)=(.*)$} $inputline allmatch attribute value] == 1} {
	    if { $currentsection != "" } {
		if { $def } {
		    iniSetDefault $currentsection $attribute $value
		} else {
		    iniSetAttribute $currentsection $attribute $value
		}
	    }
	}
    }
}

########################################################################
# iniRead() -	Read in the ini file as specified in the attributes.
#               If lines are read outside of sections, they are IGNORED!
#               0 is returned if it was NOT read, 1 otherwise.  If the
#               inifile is not read, that means either that the name of
#               the file is not known, or that the filename wasn't valid.
########################################################################
proc iniRead {} {

    if { [set iniFileName [iniGetAttribute "Program" "iniFile"]] != "" &&
         [catch {set inputfile [open $iniFileName "r"]}] == 0 } {

	     iniReadChannel $inputfile 0
	     close $inputfile
    }

    if { [set iniFileName [iniGetAttribute "Program" "tipsFile"]] != "" &&
         [catch {set inputfile [open $iniFileName "r"]}] == 0 } {

	     iniReadChannel $inputfile 1
	     close $inputfile
    }

    return 1
}
